/*
 * Toolkit GUI, an application built for operating pinkRF's signal generators.
 *
 * Contact: https://www.pinkrf.com/contact/
 * Copyright © 2018-2024 pinkRF B.V
 * GNU General Public License version 3.
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 * You should have received a copy of the GNU General Public License along with this program. If not, see https://www.gnu.org/licenses/
 *
 * Author: Iordan Svechtarov
 */

#include "serial_v3.h"
#include "miscellaneous.h"
#include "config_handler.h"
#include <QtSerialPort/QSerialPort>
#include <QElapsedTimer>
#include <QDebug>

/* Constructor */
Serial::Serial()
{
	serial = new QSerialPort();

	ConfigHandler config(QCoreApplication::applicationDirPath() + "/config.txt");
	console_output_enabled = config.get_console_output_enabled();
}

/**********************************************************************************************************************************************************************************
 * MISCELLANEOUS FUNCTIONS
 *********************************************************************************************************************************************************************************/
/* Setup the serial connection settings */
void Serial::setup(QString portname, QString intrasys_num)
{
	intrasys_number = intrasys_num;
	serial->setPortName(portname);	//COM PORT
	serial->setBaudRate(QSerialPort::Baud115200);
	serial->setDataBits(QSerialPort::Data8);
	serial->setParity(QSerialPort::NoParity);
	serial->setStopBits(QSerialPort::OneStop);
	serial->setFlowControl(QSerialPort::NoFlowControl);

	//
	// TODO:
	// Determine whether this is needed.
	//
	serial->setReadBufferSize(63);
}

bool Serial::open()
{
	if (serial->isOpen())
	{
		close();
	}

	if (serial->open(QIODevice::ReadWrite))
	{
		qDebug() << "Subsystem[" << intrasys_number.toInt() << "]: Serial ->" << serial->portName() << "Connection Established";
		return true;
	}
	qDebug() << "Subsystem[" << intrasys_number.toInt() << "]: Serial ->" << serial->error() << ":" << serial->errorString();
	return false;
}

void Serial::close()
{
	serial->close();
	qDebug() << "Subsystem[" << intrasys_number.toInt() << "]: Serial ->" << serial->portName() << "closed";
}

/* Frequently used function for sending a line over serial and receiving a reply in one go.*/
QString Serial::writeRead(QString tx)
{
	QString rx = "";
	if (serial->isOpen())
	{
		/* Check if there's anything left in the read Buffer and clean it up */
		serialClearBuffer();

		/* If missing, add newline to outbound message so that signal generator will recognize the command */
		if (!tx.contains("\r"))
		{
			tx.append("\r");
		}
		if (!tx.contains("\n"))
		{
			tx.append("\n");
		}

		/* write message to serialport */
		serialWrite(tx);
		serialRead(rx);
	}

//	if (console_output_enabled == true)
//	{
//		qDebug() << "RX:\t" << rx;
//	}

	return rx;
}


/* WriteRead until OK\r\n with flexible timeout period
 * For commands with multi-line responses like $ST,0,1 or like $SWP that can also take quite some time to complete. */
QString Serial::writeReadOK(QString tx, int timeout_ms)
{
	//
	// TODO:
	// Surely this can be combined into one with regular writeRead() somehow, no?
	// It's so similar, it's kind of stupid that this function even exists...
	//

	QString rx = "";
	if (serial->isOpen())
	{
		/* Check if there's anything left in the read Buffer and clean it up */
		serialClearBuffer();

		/* If missing, add newline to outbound message so that signal generator will recognize the command */
		if (!tx.contains("\r"))
		{
			tx.append("\r");
		}
		if (!tx.contains("\n"))
		{
			tx.append("\n");
		}

		/* write message to serialport */
		serialWrite(tx);
		serialReadOK(rx, timeout_ms);
	}

//	if (console_output_enabled == true)
//	{
//		qDebug() << "RX:\t" << rx;
//	}

	return rx;
}


/* Clear the serial buffer */
void Serial::serialClearBuffer()
{
	/* Check the read buffer for potential leftovers (for example reset related information data dump) */
	if (serial->bytesAvailable() > 0)
	{
		QString scavenge_RX;

		while(serial->waitForReadyRead(500))	//Escape the while loop if nothing comes into the buffer for 500ms.
		{
			scavenge_RX += QString::fromUtf8(serial->readAll());
		}

		scavenge_RX += QString::fromUtf8(serial->readAll());
		qDebug() << "BUFFER JUNK: " << scavenge_RX;
	}
}


/* Write a message to the serial buffer */
void Serial::serialWrite(QString tx)
{
	/* Display the message to transmit if requested */
	if (console_output_enabled == true)
	{
		qDebug() << "TX:\t" << tx;
	}

	if(serial->isOpen())
	{
		serial->write((const char*)(tx).toUtf8());
		serial->waitForBytesWritten(250);
	}
}


/* Read from the serial bus */
bool Serial::serialRead(QString& rx, int timeout_ms)
{
	//
	// TODO:
	// UART specific issue:
	// A badly-timed reset can result in string concatenation with the EEPROM dump.
	// This can result in a broken string such as "$DLCG,1,2400.000000,2500.000000,245$CHANS,1,OK\r\n"
	// This unfortunately passes all sanity checking (starts with "$DLCG," && ends on "\r\n" && excludes ",ERR"), and can cause a crash once a function starts processing it.
	// The recommended solution is to remove the EEPROM dump from the firmware entirely at this point.
	//

	QElapsedTimer timer;
	timer.start();

	/* Do at least once */
	do
	{
		serial->waitForReadyRead(1);

		while (serial->bytesAvailable() > 0)	//Solves a timing-related issue of packages getting lost, for example when doing a particularly granular sweep such as "$SWP,0,2400,2500,0.1,100,0"
		{
			char buf[serial->bytesAvailable()+2];	//maxSize must be larger than 1
			serial->readLine(buf, sizeof (buf));
			rx += buf;
		}
	}
	while(!rx.endsWith("\r\n") && timer.elapsed() < timeout_ms);

	if (console_output_enabled == true)
	{
		qDebug() << "RX:\t" << rx;
	}

	if (timer.elapsed() >= timeout_ms)
	{
//		qDebug() << timer.elapsed();
		return false;
	}

	return true;
}

/* Keep Read until OK string or timeout */
bool Serial::serialReadOK(QString& rx, int timeout_ms)
{
	//
	// TODO:
	// UART specific issue:
	// A badly-timed reset can result in string concatenation with the EEPROM dump.
	// This can result in a broken string such as "$DLCG,1,2400.000000,2500.000000,245$CHANS,1,OK\r\n"
	// This unfortunately passes all sanity checking (starts with "$DLCG," && ends on "\r\n" && excludes ",ERR"), and can cause a crash once a function starts processing it.
	// The recommended solution is to remove the EEPROM dump from the firmware entirely at this point.
	//

	QElapsedTimer timer;
	timer.start();

	/* Do at least once.
	 * Stop when finished, errored or timed out */
	do
	{
		serial->waitForReadyRead(1);

		while (serial->bytesAvailable() > 0)	//Solves a timing-related issue of packages getting lost, for example when doing a particularly granular sweep such as "$SWP,0,2400,2500,0.1,100,0"
		{
			char buf[serial->bytesAvailable()+2];	//maxSize must be larger than 1
			serial->readLine(buf, sizeof (buf));
			rx += buf;
		}
	}
	while ( !rx.endsWith(",OK\r\n") && !(rx.contains(",ERR") && rx.endsWith("\r\n")) && timer.elapsed() < timeout_ms );

	if (console_output_enabled == true)
	{
		qDebug() << "RX:\t" << rx;
	}

	if (timer.elapsed() >= timeout_ms)
	{
//		qDebug() << timer.elapsed();
		return false;
	}

	return true;
}


/**********************************************************************************************************************************************************************************
 * Transparent Mode
 * *******************************************************************************************************************************************************************************/
/* Write a message to the serial buffer
 * If something is wrong with the serial port, try closing and re-opening it */
void Serial::RCM_Write(QString tx)
{
	if (serial->error() != QSerialPort::NoError)
	{
		qDebug() << serial->error() << serial->errorString();
		serial->close();
		serial->open(QIODevice::ReadWrite);
	}

	if(serial->isOpen())
	{
		serial->write((const char*)(tx).toUtf8());
//		serial->waitForBytesWritten(250);
	}
}

/* Read from the serial buffer
 * Pretty much identical to serialRead but doesn't spawn buffer fill-up messages */
QString Serial::RCM_Read()
{
	QString rx = "";
	if (serial->isOpen())
	{
		rx += QString::fromUtf8(serial->readAll());
	}

	return rx;
}
